from django.core.management.base import BaseCommand
from django.contrib.sites.models import Site
from allauth.socialaccount.models import SocialApp
from django.db import IntegrityError


class Command(BaseCommand):
    help = 'Setup Google OAuth configuration'

    def add_arguments(self, parser):
        parser.add_argument('--client-id', type=str, help='Google Client ID')
        parser.add_argument('--client-secret', type=str, help='Google Client Secret')

    def handle(self, *args, **options):
        self.stdout.write(self.style.SUCCESS('=== Google OAuth Setup ===\n'))
        
        # Update or create site
        try:
            site = Site.objects.get(id=1)
            site.domain = '127.0.0.1:8000'
            site.name = 'Tayapromediverse'
            site.save()
            self.stdout.write(self.style.SUCCESS(f'✓ Updated site: {site.domain}'))
        except Site.DoesNotExist:
            site = Site.objects.create(
                id=1,
                domain='127.0.0.1:8000',
                name='Tayapromediverse'
            )
            self.stdout.write(self.style.SUCCESS(f'✓ Created site: {site.domain}'))
        
        # Get credentials from command line or prompt
        client_id = options.get('client_id')
        client_secret = options.get('client_secret')
        
        if not client_id:
            client_id = input('\nEnter Google Client ID: ').strip()
        
        if not client_secret:
            client_secret = input('Enter Google Client Secret: ').strip()
        
        if not client_id or not client_secret:
            self.stdout.write(self.style.ERROR('\n✗ Both Client ID and Client Secret are required!'))
            self.stdout.write(self.style.WARNING('\nTo get credentials:'))
            self.stdout.write('1. Go to https://console.cloud.google.com/')
            self.stdout.write('2. Create OAuth credentials')
            self.stdout.write('3. Run this command again with credentials\n')
            return
        
        # Create or update Google Social App
        try:
            social_app = SocialApp.objects.get(provider='google')
            social_app.name = 'Google OAuth'
            social_app.client_id = client_id
            social_app.secret = client_secret
            social_app.save()
            self.stdout.write(self.style.SUCCESS('✓ Updated Google OAuth app'))
        except SocialApp.DoesNotExist:
            social_app = SocialApp.objects.create(
                provider='google',
                name='Google OAuth',
                client_id=client_id,
                secret=client_secret,
            )
            self.stdout.write(self.style.SUCCESS('✓ Created Google OAuth app'))
        
        # Link the social app to the site
        if site not in social_app.sites.all():
            social_app.sites.add(site)
            self.stdout.write(self.style.SUCCESS(f'✓ Linked Google OAuth to site: {site.domain}'))
        
        self.stdout.write(self.style.SUCCESS('\n=== Setup Complete! ==='))
        self.stdout.write(self.style.SUCCESS('\nYou can now use Google Sign-In at:'))
        self.stdout.write(f'  http://{site.domain}/signin/\n')
        self.stdout.write(self.style.WARNING('Note: Make sure your Google Console redirect URI is:'))
        self.stdout.write(f'  http://{site.domain}/accounts/google/login/callback/\n')

