#!/bin/bash

# =============================================================================
# Django Server Setup Script
# Run this script ON THE SERVER after uploading files
# =============================================================================

echo "🚀 Starting Django deployment setup..."
echo ""

# Colors for output
GREEN='\033[0;32m'
BLUE='\033[0;34m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# =============================================================================
# Step 1: Check Current Directory
# =============================================================================

echo -e "${BLUE}📍 Step 1: Checking directory...${NC}"
pwd
echo ""

# =============================================================================
# Step 2: Create Virtual Environment
# =============================================================================

echo -e "${BLUE}🐍 Step 2: Creating virtual environment...${NC}"

# Try to find Python 3
if command -v python3.9 &> /dev/null; then
    PYTHON_CMD="python3.9"
elif command -v python3.8 &> /dev/null; then
    PYTHON_CMD="python3.8"
elif command -v python3 &> /dev/null; then
    PYTHON_CMD="python3"
else
    echo -e "${RED}❌ Error: Python 3 not found!${NC}"
    exit 1
fi

echo "Using Python: $PYTHON_CMD"
$PYTHON_CMD --version
echo ""

# Create virtual environment
if [ ! -d "venv" ]; then
    $PYTHON_CMD -m venv venv
    echo -e "${GREEN}✅ Virtual environment created${NC}"
else
    echo -e "${GREEN}✅ Virtual environment already exists${NC}"
fi
echo ""

# =============================================================================
# Step 3: Activate Virtual Environment
# =============================================================================

echo -e "${BLUE}⚡ Step 3: Activating virtual environment...${NC}"
source venv/bin/activate
echo -e "${GREEN}✅ Virtual environment activated${NC}"
echo ""

# =============================================================================
# Step 4: Upgrade pip
# =============================================================================

echo -e "${BLUE}📦 Step 4: Upgrading pip...${NC}"
pip install --upgrade pip
echo ""

# =============================================================================
# Step 5: Install Requirements
# =============================================================================

echo -e "${BLUE}📚 Step 5: Installing requirements...${NC}"
echo "⏳ This may take 5-10 minutes..."
pip install -r requirements.txt
echo -e "${GREEN}✅ Requirements installed${NC}"
echo ""

# =============================================================================
# Step 6: Create Required Directories
# =============================================================================

echo -e "${BLUE}📁 Step 6: Creating directories...${NC}"
mkdir -p logs
mkdir -p staticfiles
mkdir -p mediafiles
echo -e "${GREEN}✅ Directories created${NC}"
echo ""

# =============================================================================
# Step 7: Run Migrations
# =============================================================================

echo -e "${BLUE}🗄️  Step 7: Running database migrations...${NC}"
python manage.py migrate --settings=conn_prjct.settings_production
echo -e "${GREEN}✅ Migrations completed${NC}"
echo ""

# =============================================================================
# Step 8: Collect Static Files
# =============================================================================

echo -e "${BLUE}🎨 Step 8: Collecting static files...${NC}"
python manage.py collectstatic --noinput --settings=conn_prjct.settings_production
echo -e "${GREEN}✅ Static files collected${NC}"
echo ""

# =============================================================================
# Step 9: Set Permissions
# =============================================================================

echo -e "${BLUE}🔒 Step 9: Setting file permissions...${NC}"
chmod -R 755 .
chmod 644 db.sqlite3 2>/dev/null || echo "db.sqlite3 not found yet"
chmod 755 staticfiles/
chmod 755 mediafiles/
echo -e "${GREEN}✅ Permissions set${NC}"
echo ""

# =============================================================================
# Step 10: Test Django Configuration
# =============================================================================

echo -e "${BLUE}🧪 Step 10: Testing Django configuration...${NC}"
python manage.py check --settings=conn_prjct.settings_production
echo ""

# =============================================================================
# Step 11: Create Superuser
# =============================================================================

echo -e "${BLUE}👤 Step 11: Create superuser...${NC}"
echo "You'll be prompted to create an admin account."
echo "Press Enter to continue, or Ctrl+C to skip..."
read -p ""
python manage.py createsuperuser --settings=conn_prjct.settings_production
echo ""

# =============================================================================
# Final Summary
# =============================================================================

echo ""
echo -e "${GREEN}═══════════════════════════════════════════════════════${NC}"
echo -e "${GREEN}✅ Setup Complete!${NC}"
echo -e "${GREEN}═══════════════════════════════════════════════════════${NC}"
echo ""
echo -e "${BLUE}📝 Next Steps:${NC}"
echo "1. Configure Python App in cPanel"
echo "2. Set up SSL certificate"
echo "3. Test your website"
echo ""
echo -e "${BLUE}📍 Important Paths:${NC}"
echo "   Application Root: $(pwd)"
echo "   Virtual Environment: $(pwd)/venv"
echo "   Static Files: $(pwd)/staticfiles"
echo "   Media Files: $(pwd)/mediafiles"
echo ""
echo -e "${BLUE}🔧 To manually activate virtual environment later:${NC}"
echo "   cd $(pwd)"
echo "   source venv/bin/activate"
echo ""
echo -e "${GREEN}🎉 Your Django app is ready for deployment!${NC}"
echo ""


