# Troubleshooting Fixed

## Issue: Login and Signup Pages Not Working

### Symptoms

- Login page (`/signin/`) not loading properly
- Signup page (`/`) not loading properly
- Google OAuth throwing `MultipleObjectsReturned` error

### Root Causes Found

1. **Multiple Sites in Database**

   - Problem: Two `Site` objects existed (ID 1 and ID 2)
   - Impact: django-allauth couldn't determine which site to use
   - Solution: Deleted duplicate site, kept only Site ID 1

2. **Conflicting Google OAuth Configuration**
   - Problem: `APP` configuration in `SOCIALACCOUNT_PROVIDERS` (settings.py) conflicted with database `SocialApp`
   - Impact: MultipleObjectsReturned error at `/accounts/google/login/`
   - Solution: Removed `APP` dict from settings.py, use only database configuration

### Fixes Applied

#### 1. Fixed Duplicate Sites

```bash
python manage.py fix_sites
```

What it does:

- Deletes all sites except Site ID 1
- Updates Site ID 1 with correct domain (127.0.0.1:8000)
- Ensures only one site exists

#### 2. Fixed Google OAuth Configuration

**Before** (settings.py):

```python
SOCIALACCOUNT_PROVIDERS = {
    'google': {
        'SCOPE': ['profile', 'email'],
        'AUTH_PARAMS': {'access_type': 'online'},
        'APP': {
            'client_id': '',
            'secret': '',
            'key': ''
        }
    }
}
```

**After** (settings.py):

```python
SOCIALACCOUNT_PROVIDERS = {
    'google': {
        'SCOPE': ['profile', 'email'],
        'AUTH_PARAMS': {'access_type': 'online'}
    }
}
```

Google OAuth credentials are now managed only through:

1. Django Admin → Social applications
2. Command: `python manage.py setup_google_oauth`

### Verification

All pages now working correctly:

- ✅ Home/Signup: http://127.0.0.1:8000/ (200 OK)
- ✅ Sign In: http://127.0.0.1:8000/signin/ (200 OK)
- ✅ Dashboard: http://127.0.0.1:8000/dashboard/ (200 OK)
- ✅ Subscription: http://127.0.0.1:8000/subscription/ (200 OK)
- ✅ Google OAuth: http://127.0.0.1:8000/accounts/google/login/ (302 Redirect)

### Diagnostic Commands Created

Three management commands were created to help debug similar issues:

1. **check_sites** - Check Sites and SocialApp configuration

   ```bash
   python manage.py check_sites
   ```

2. **fix_sites** - Fix duplicate Sites automatically

   ```bash
   python manage.py fix_sites
   ```

3. **debug_google_oauth** - Detailed Google OAuth debug info

   ```bash
   python manage.py debug_google_oauth
   ```

4. **fix_google_oauth** - Fix duplicate Google OAuth apps
   ```bash
   python manage.py fix_google_oauth
   ```

### How to Avoid This in Future

1. **Single Site Per Environment**

   - Keep only one Site object per environment
   - Use Site ID 1 for local development
   - Use different Site IDs for staging/production

2. **Google OAuth Configuration**

   - Store credentials in database via Django Admin or management command
   - Don't use `APP` dict in `SOCIALACCOUNT_PROVIDERS` settings
   - Settings should only contain SCOPE and AUTH_PARAMS

3. **Best Practices**
   - Use `settings.SITE_ID = 1` to explicitly set the site
   - Verify configuration with diagnostic commands before deployment
   - Keep environment-specific settings in separate files or .env

### Testing Google Sign-In

To fully test Google OAuth:

1. Get real Google OAuth credentials from https://console.cloud.google.com/
2. Run setup command:
   ```bash
   python manage.py setup_google_oauth --client-id YOUR_ID --client-secret YOUR_SECRET
   ```
3. Visit http://127.0.0.1:8000/signin/
4. Click "Continue with Google"
5. Should redirect to Google login ✓

### Status

**Current Status**: ✅ ALL WORKING

- Regular login/signup: ✅ Working
- Google OAuth setup: ✅ Ready (needs real credentials to test)
- Subscription system: ✅ Working
- All pages loading: ✅ Working

---

**Date Fixed**: October 26, 2025
**Fixed By**: AI Assistant
