from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.contrib import messages
import json
from django.http import JsonResponse
from django.utils import timezone
from django.core.paginator import Paginator
from django.views.decorators.http import require_POST

from .models import Profile
from .forms import ProfileUpdateForm, SecuritySettingsForm, NameUpdateForm, PasswordChangeForm

# -------------------------
# PROFILE VIEWS
# -------------------------
@login_required
def profile_view(request):
    if request.method == 'POST':
        form = ProfileUpdateForm(request.POST, request.FILES, instance=request.user.profile)
        if form.is_valid():
            # Update user info
            user = request.user
            user.first_name = form.cleaned_data['name']
            
            # Check if email changed
            new_email = form.cleaned_data['email']
            if new_email != user.email:
                user.email = new_email
                user.username = new_email  # Since we use email as username
            
            user.save()
            form.save()
            
            messages.success(request, "Profile updated successfully!")
            return redirect('profile')
    else:
        form = ProfileUpdateForm(instance=request.user.profile)
    
    return render(request, 'profile.html', {'form': form})

@login_required
def security_settings_view(request):
    name_form = NameUpdateForm(request.user)
    password_form = PasswordChangeForm(request.user)
    
    if request.method == 'POST':
        form_type = request.POST.get('form_type')
        
        if form_type == 'name':
            name_form = NameUpdateForm(request.user, request.POST)
            if name_form.is_valid():
                user = request.user
                user.first_name = name_form.cleaned_data['first_name']
                user.last_name = name_form.cleaned_data['last_name']
                user.save()
                messages.success(request, "Name updated successfully!")
                return redirect('settings')
                
        elif form_type == 'password':
            password_form = PasswordChangeForm(request.user, request.POST)
            if password_form.is_valid():
                user = request.user
                user.set_password(password_form.cleaned_data['new_password'])
                user.save()
                
                # Update the session to keep the user logged in
                from django.contrib.auth import update_session_auth_hash
                update_session_auth_hash(request, user)
                
                messages.success(request, "Password changed successfully!")
                return redirect('settings')
    
    context = {
        'name_form': name_form,
        'password_form': password_form,
        'user': request.user,
    }
    return render(request, 'setting.html', context)

# -------------------------
# NOTIFICATION VIEWS (DISABLED)
# -------------------------
@login_required
def notifications_view(request):
    messages.info(request, "Notifications functionality is currently disabled.")
    return redirect('profile')

@login_required
@require_POST
def mark_notification_read(request, notification_id):
    return JsonResponse({'success': False, 'error': 'Notifications disabled'})

@login_required
@require_POST
def mark_all_notifications_read(request):
    return JsonResponse({'success': False, 'error': 'Notifications disabled'})

@login_required
@require_POST
def update_notification_preferences(request):
    return JsonResponse({'success': False, 'error': 'Notifications disabled'})